# -------------------------------------------------------------------------------------
# flo::plugin::Admin::Properties
# -------------------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver
#    Copyright : (c) MKDoc Holdings Ltd, 2002
#
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#    Description:
#
# -------------------------------------------------------------------------------------
package flo::plugin::Admin::Properties;
use MKDoc::Util::TemplateFinder;
use MKDoc::Config;
use MKDoc::Error;
use flo::Standard;
use flo::RedirectManager;
use strict;
use warnings;
use 5.008_000;
use Carp;

use base qw /flo::Plugin/;

sub template_path { '/admin/properties' }


##
# $class->templates;
# ------------------
# Returns a structure which can be used within a template to choose
# a document template from a select box.
#
# See MKDoc::Util::TemplateFinder::list_structure()
##
sub templates
{
    my $self    = shift;
    my $default = $self->parent()->template() || shift || 'default';
    return MKDoc::Util::TemplateFinder::list_structure ('document', $default);
}


##
# $self->selected_template;
# -------------------------
# Returns the template which is currently selected.
#
# See MKDoc::Util::TemplateFinder::list_structure_selected()
##
sub selected_template
{
    my $self = shift;
    my $selected = MKDoc::Util::TemplateFinder::list_structure_selected ($self->templates());
    $selected ||= do {
	my $name  = $self->parent->popular_children_template;
	my $label = $name;
	$label =~ s/_/ /g;
	{
	    value    => $name,
	    label    => $label,
	    selected => 'selected',
	};
    };
    
    return $selected;
}


##
# $self->unselected_templates;
# ----------------------------
# Returns the templates which are not currently selected.
#
# See MKDoc::Util::TemplateFinder::list_structure_unselected()
##
sub unselected_templates
{
    my $self = shift;
    return MKDoc::Util::TemplateFinder::list_structure_unselected ($self->templates());
}


##
# $class->cache_controls;
# -----------------------
#   Returns a structure which can be used within a template to choose
#   a document cache control value from a select box.
##
sub cache_controls
{
    my $self = shift;
    my $doc  = flo::Standard::current_document();
    my $cgi  = flo::Standard::cgi();
    
    my $selected_cache_control = $cgi->param ('cache_control');
    $selected_cache_control    = $doc->cache_control unless (defined $selected_cache_control);
    
    my @res = map {
	{
	    value    => $_,
	    label    => $_,
	    selected => ($_ == $selected_cache_control),
	}
    } $doc->available_cache_controls;
    
    return wantarray ? @res : \@res;
}


##
# $self->selected_cache_control;
# ------------------------------
#   Returns the cache_control which is currently selected
##
sub selected_cache_control
{
    my $self = shift;
    my ($selected) = map { ($_->{selected}) ? $_ : () } $self->cache_controls (@_);
    $selected ||= do {
	{
	    value => $self->cache_control,
	    label => $self->cache_control,
	    selected => 'selected',
	};
    };
    
    return $selected;
}


##
# $self->unselected_cache_controls;
# ---------------------------------
#   Returns the cache_controls which are currently not selected
##
sub unselected_cache_controls
{
    my $self = shift;
    $self = ref $self || $self;
    my @unselected = map { ($_->{selected}) ? () : $_ } $self->cache_controls (@_);
    return (wantarray) ? @unselected : \@unselected;
}


##
# $class->langs;
# --------------
#   Returns a structure which can be used within a template to choose
#   a document cache control value from a select box.
##
sub langs
{
    my $self = shift;
    my $doc  = $self->parent;
    my $cgi  = flo::Standard::cgi();
    
    my $current_lang = $cgi->param ('lang') || $doc->lang;
    my $languages_rtl = MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST_RTL);
    my @res = map {
	{
	    value          => $_,
	    label          => $doc->lang_label ($_),
	    selected       => ($_ eq $current_lang),
	    direction      => (exists $languages_rtl->{$_}) ? 'rtl'   : 'ltr',
	    align          => (exists $languages_rtl->{$_}) ? 'right' : 'left',
	    align_opposite => (exists $languages_rtl->{$_}) ? 'left'  : 'right',
	}
    } $doc->available_langs;
    
    return wantarray ? @res : \@res;
}


##
# $self->selected_cache_control;
# ------------------------------
#   Returns the language which is currently selected
##
sub selected_lang
{
    my $self = shift;
    my ($selected) = map { ($_->{selected}) ? $_ : () } $self->langs (@_);
    $selected ||= do {
	my $languages_rtl = MKDoc::Config->parsefile_hashref (MKDoc::Config->LANGUAGE_LIST_RTL) ||
	    die 'Could not retrieve language list!';
	
	my $val = $self->parent->lang;
	{
	    value => $val,
	    label => $self->parent->lang_label ($val),
	    selected => 'selected',
	    direction      => (exists $languages_rtl->{$val}) ? 'rtl'   : 'ltr',
	    align          => (exists $languages_rtl->{$val}) ? 'right' : 'left',
	    align_opposite => (exists $languages_rtl->{$val}) ? 'left'  : 'right',
	}
    };

    return $selected;
}


##
# $self->unselected_langs;
# ------------------------
#   Returns the langs which are currently not selected
##
sub unselected_langs
{
    my $self = shift;
    $self = ref $self || $self;
    my @unselected = map { ($_->{selected}) ? () : $_ } $self->langs (@_);
    return (wantarray) ? @unselected : \@unselected;
}

##
# $self->groups;
# -----------------
#   Returns all groups related to the current document,
#   but adds a 'checked' flag for those which are checked and a 
#   'readonly' flag for those that should not be editable.
##
sub groups {
    my $self = shift;
    my $dbh = lib::sql::DBH->get();
    my $results = $dbh->selectall_arrayref(
      'SELECT ID, Name, Description FROM Grp ORDER BY name');

    my @groups;
    foreach my $row (@$results) {
        my %group = (id          => $row->[0],
                     name        => $row->[1],
                     description => $row->[2]);
        $self->_group_selection(\%group);
        push(@groups, \%group);
    }

    return wantarray ? @groups : \@groups;
}

# determine whether a group is selected, marking it readonly if the
# selection comes from a parent.  Also mark readonly if the editor
# isn't a member of the group.
sub _group_selection {
    my ($self, $group) = @_;
    my $dbh            = lib::sql::DBH->get();
    my $document       = flo::Standard::current_document();
    my $group_id       = $group->{id};

    # get list of all documents to check
    my @documents = ($document, $document->ancestors);
    
    # get results for each document for this group
    my $this_id    = $document->id;
    foreach my $doc (@documents) {        
        my ($exists) = $dbh->selectrow_array(
          'SELECT 1 FROM Document_Grp WHERE Document_ID = ? AND Grp_ID = ?',
          undef, $doc->id, $group_id);
        
        # got a winner, set it checked and reasonly if this came from
        # up the line
        if ($exists) {
            $group->{checked}  = 1;
            $group->{readonly} = 1 unless $doc->id == $this_id;
            last;
        }
    }

    # check for group membership if the group isn't already readonly
    unless ($group->{readonly}) {
        my ($member) = $dbh->selectrow_array(
          'SELECT 1 FROM Editor_Grp WHERE Editor_ID = ? AND Grp_ID = ?',
          undef, $self->user->id, $group_id);
        $group->{readonly} = 1 unless $member;
    }
}

    


##
# $self->audiences;
# -----------------
#   Returns all audiences related to the current document,
#   but adds a 'checked' flag for those which are checked.
##
sub audiences
{
    my $self = shift;
    my $document = flo::Standard::current_document();
    
    my $checked = { map { $_->id() => 1 } $document->audiences };
    my $audience_t = flo::Standard::table ('Audience');
    my @audiences = $audience_t->select ( sort => [ qw /Label/ ] )->fetch_all;
    $_->{checked} = ($checked->{$_->id()}) ? 'checked' : undef for (@audiences);
    
    return wantarray ? @audiences : \@audiences;
}


##
# $self->audience_categories();
# ------------------------------------
#   Returns a sorted list of all audience categories.
#   Categories are implicit when a 'label' has ':' in it.
#   Otherwise items will be in category ''.
##
sub audience_categories
{
    my $self = shift;

    my $categories;
    for my $audience (@{$self->audiences})
    {
        $categories->{$audience->category} = undef;
    }

    my @audience_categories;
    for (keys %{$categories}) { push @audience_categories, $_ };

    @audience_categories = sort { $a cmp $b } @audience_categories;

    return wantarray ? @audience_categories : [@audience_categories];
}


##
# $self->audiences_by_category ('Fruit');
# ------------------------
#   Returns a list of all flo::Record::Audience objects available
#   to be set that match the specified category, ordered by label, or undef on error.
##
sub audiences_by_category
{
    my $self = shift;
    my $category = shift;

    my @AUDIENCES;
    my @AUDIENCE_BY_CATEGORY;

    my @auds   = $self->audiences;
    @AUDIENCES = sort { $a->label cmp $b->label } @auds;
    for (@AUDIENCES)
    {
        push @AUDIENCE_BY_CATEGORY, $_ if $_->category eq $category;
    }

    return wantarray ? @AUDIENCE_BY_CATEGORY : [@AUDIENCE_BY_CATEGORY];
}


##
# $self->activate;
# ----------------
#   Returns TRUE if this plugin can be activated, FALSE
#   otherwise.
##
sub activate
{
    my $self = shift;
    return unless ($self->SUPER::activate (@_));
    my $current_document = flo::Standard::current_document();
    return $current_document->Can_Admin;
}


##
# $self->http_get;
# ----------------
# Displays the form which lets the editor choose which audiences
# the current document relates to.
##
sub http_get
{
    my $self = shift;
    $self->render_http (
	self       => $self,
	object     => flo::Standard::current_document(),
	__input__  => 'XML',
	__output__ => 'XHTML',
       );
    
    return 'TERMINATE';
}


##
# $self->http_post;
# -----------------
#   Reflect the changes on the database and redirects to
#   current document.
##
sub http_post
{
    my $self = shift;
    my $cgix = flo::Standard::cgi()->new;
    
    my $document_t = flo::Standard::table ('Document');
    my $document = $self->parent;
    
    # let's set the document object attributes
    $document->set_title         ($cgix->param ( 'title'         ) );
    $document->set_description   ($cgix->param ( 'description'   ) );
    $document->set_keywords      ($cgix->param ( 'keywords'      ) );
    $document->set_lang          ($cgix->param ( 'lang'          ) );
    $document->set_rights        ($cgix->param ( 'rights'        ) );
    $document->set_publisher     ($cgix->param ( 'publisher'     ) );
    $document->set_template      ($cgix->param ( 'template'      ) );
    $document->set_cache_control ($cgix->param ( 'cache_control' ) );
    
    # update date last modified to now
    $document->set_date_last_modified ($document->now_iso());
    
    # we want to set the document name unless it's the root document
    if ($document->parent)
    {
	my $name = $cgix->param ( 'name' );
	flo::RedirectManager->document_renamed ($document, $name);
	$document->set_name ($name);
    }
    
    $document->validate() || return $self->http_get();
    $document->save();
    $self->_set_audiences_from_cgi ($document);
    $self->_set_groups_from_cgi ($document);
    
    # we have finished changing the properties, let's redirect
    # to the current document
    $cgix->delete ($_) for ($cgix->param);
    $cgix->path_info ($document->path);
    print $cgix->redirect ($cgix->self_url);    
    return 'TERMINATE';
}


##
# $self->_set_audiences_from_cgi;
# -------------------------------
#   Sets the audiences in the 'Audience' table from the CGI object.
##
sub _set_audiences_from_cgi
{
    my $self = shift;
    my $document = shift;
    my $cgix = flo::Standard::cgi->new;
    
    my $document_audience_table = flo::Standard::table ('Document_Audience');
    my $audience_table = flo::Standard::table ('Audience');
    my @submited_audiences = $cgix->param ('audience');
    
    my %existing_audiences = map { $_->id() => 1 } $audience_table->search()->fetch_all();
    my @valid_ids = ();
    foreach my $audience_id (@submited_audiences)
    {
	next unless $existing_audiences{$audience_id};
	push @valid_ids, $audience_id;
    }
    
    $document_audience_table->delete ( Document_ID => $document->id() );
    $document_audience_table->insert ( Document_ID => $document->id(), Audience_ID => $_ ) for (@valid_ids);
}

##
# $self->_set_groups_from_cgi;
# -------------------------------
#   Sets the groups in the 'Document_Grp' table from the CGI object.
##
sub _set_groups_from_cgi {
    my $self        = shift;
    my $document    = shift;
    my $document_id = $document->id;
    my $cgi         = flo::Standard::cgi->new;
    my $dbh         = lib::sql::DBH->get();    

    my %checked = map { $_ => 1 } $cgi->param('group');
    my %groups  = map { $_->{id} => $_ } $self->groups();
    foreach my $group_id (keys %groups) {
        # don't touch readonly settings
        next if $groups{$group_id}->{readonly};

        # clear the current value
        $dbh->do('DELETE FROM Document_Grp
                  WHERE Document_ID = ? AND Grp_ID = ?',
                 undef, $document_id, $group_id);

        # set if checked
        $dbh->do('INSERT INTO Document_Grp (Document_ID, Grp_ID) VALUES (?,?)',
                 undef, $document_id, $group_id)
          if $checked{$group_id};
    }
}


##
# $self->lang;
# ------------
#   Returns the currently selected language,
#   or the language of the parent if none is selected.
##
sub lang
{
    my $self = shift;
    my $current = $self->selected_lang;
    return $current->{value} if (defined $current);
    return $self->SUPER::lang;
}


##
# $self->form_input ($name);
# --------------------------
# Returns the default value to use for the form of
# name $name.
##
sub form_input
{
    my $self   = shift;
    my $param  = shift;
    my $object = flo::Standard::current_document();
    my $cgix   = flo::Standard::cgi()->new();
    return $cgix->param ($param) || $object->$param();
}


1;
