# ------------------------------------------------------------------------
# flo::plugin::Comment
# ------------------------------------------------------------------------
# Author : Bruno Postle.
# Copyright : (c) MKDoc Ltd, 2005.
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ------------------------------------------------------------------------
package flo::plugin::Comment;
use flo::Standard;
use flo::Plugin;
use flo::editor::Comment;
use Petal::Mail;
use strict;
use warnings;
use 5.008_000;
use utf8;

use vars qw /@ISA/;
@ISA = qw /flo::Plugin/;

=head1 NAME

flo::plugin::Comment - ability for any authenticated user to add comments

=head1 SYNOPSIS

Access this plugin like this:

  http://www.example.com/.comment.html
  http://www.example.com/foo/.comment.html

=head1 DESCRIPTION

This plugin implements a simple form that accepts plain text.  When submitted,
and so long as it isn't empty, this text gets appended to to the end of the
document as the content of a L<flo::editor::Comment> component.

This plugin doesn't control the later display or any other usage that the
component may be put to.

Once the form is POSTed, the user is redirected back to the parent document
(which may or may not display the commment itself).

The Contributor table is updated to indicate that this user has contributed to
this document

=cut

sub template_path { return '/comment' };

sub _name_default { '.comment.html' }

sub http_get
{
    my $self = shift;
    my $cgi = flo::Standard::cgi();
    $self->render_http (
	self       => $self,
	object     => $self->parent,
	__input__  => 'XML',
	__output__ => 'XHTML',
       );
    return 'TERMINATE';
}

sub http_post
{
    my $self  = shift;
    my $cgi   = $self->cgi();

    $self->_update_document_from_cgi();
    $self->_update_contributor();
    $self->{comment} = $self->cgi->param ('comment');
    $self->_send_mail();

    print $cgi->redirect ($self->parent->uri);
    return 'TERMINATE';
}

sub _update_document_from_cgi
{
    my $self   = shift;
    my $user   = $self->user;
    my $cgi    = $self->cgi;

    return unless ($cgi->param ('comment') =~ /[[:alnum:]]/);

    my $text_obj = new flo::editor::Comment (
        param_name => 'comment',
        cgi        => $cgi
    );

    my $document = $self->parent;
    $document->add_component ($text_obj);
    $document->save();
}

sub _update_contributor
{
    my $self = shift;
    my $user = $self->user;
    my $document = $self->parent;

    my $contributor_t = flo::Standard::table ('Contributor');
    $contributor_t->get (Editor_ID => $user->id, Document_ID => $document->id) and return 1;
    $contributor_t->insert (Editor_ID => $user->id, Document_ID => $document->id);
}

=pod

The plugin only activates if enabled in the plugin list and if the user is
authenticated.

=cut

sub activate
{
    my $self = shift;
    return 0 unless ($self->SUPER::activate(@_));
    return 0 unless ($self->user);
    return $self;
}


sub _send_mail
{
    my $self   = shift;
    
    my $xml_mail = eval {
        my $mail = new Petal::Mail (
            file     => '/comment/email',
            language => $self->language(),
        );

	$mail->send (
	    self       => $self,
	    object     => $self->parent(),
	    __input__  => 'XML',
	    __output__ => 'XML',
	);
    };
}

1;

__END__
