# -------------------------------------------------------------------------------------
# flo::plugin::Discussion::Post
# -------------------------------------------------------------------------------------
#
#       Author : Jean-Michel Hiver
#    Copyright : (c) MKDoc Holdings Ltd, 2002
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#    Description:
#
# -------------------------------------------------------------------------------------
package flo::plugin::Discussion::Post;
use MKDoc::Config;
use flo::ComponentPlugin;
use flo::HTTP::Header;
use flo::plugin::Discussion::IMAP;
use flo::plugin::Discussion::Message;
use flo::Standard qw /cgi table/;
use strict;
use warnings;
use 5.008_000;
use Carp;

use base qw /flo::plugin::Discussion::List/;

sub type   { 'discussion' }
sub cgi_do { 'post'       }


sub activate
{
    my $self = shift;
    return unless ($self->SUPER::_activate_passthru);
    
    my $cgi = flo::Standard::cgi();
    my $path_info = $cgi->path_info;
    
    my $hint = quotemeta (MKDoc::Config->IMAP_POST_URI_HINT);
    $path_info =~ /\,$hint$/ || return;
    
    my $component = $self->component;
    $component->isa ('flo::editor::Discussion') || return;
    
    $self->_initialize_after_activate;
    
    # if there was an error during initialization, we want to activate in order
    # to send an error page
    return 1 if ($self->{error});
    
    # we want to activate only if the message with 'uid' exists
    # in the current IMAP folder
    my $uid = $cgi->param ('uid') || undef;
    
    return if (defined $uid and $uid and $uid !~ /^\d+$/);
    
    my $imap = new flo::plugin::Discussion::IMAP();
    return if (defined $uid and not $imap->message_exists ($uid));
    
    # everything is fine, let's do a tiny bit of extra init
    $self->{template} = 'discussion/post';
    return 1;
}


##
# $class->user;
# -------------
#   Returns the current user information, undef otherwise
##
sub user
{
    return $::MKD_USER;
}


##
# $self->run;
# -----------
#   Displays a list of messages
##
sub run
{
    my $self = shift;
    my $document_table = table ('Document');
    my $cgi = cgi;
    
    if ($ENV{REQUEST_METHOD} =~ /^POST$/i) { $self->http_post }
    else                                   { $self->http_get  }
    eval
    {
	flo::plugin::Discussion::IMAP->kaboom;
	flo::plugin::Discussion::Message->kaboom;
    };
    
    return 'TERMINATE';
}


##
# $self->reply_uid;
# -----------------
#   Returns the UID of the message to reply to, if any
sub reply_uid
{
    my $self = shift;
    my $cgi = cgi();
    return $cgi->param ('uid');
}


##
# $self->reply_message;
# ---------------------
#   Returns the message object which this forum post replies to,
#   or undef if it's a new post.
##
sub reply_message
{
    my $self = shift;
    my $reply_uid = $self->reply_uid;
    my $msg_obj = new flo::plugin::Discussion::Message ($reply_uid) if (defined $reply_uid and $reply_uid);
    return $msg_obj;
}


##
# $self->subject;
# ---------------
#   Returns the default subject for this message.
##
sub subject
{
    my $self = shift;
    my $cgi  = cgi();
    my $subject = $cgi->param ('subject');
    
    my $msg_obj = $self->reply_message;
    if (not defined $subject and defined $msg_obj)
    {
	$subject ||= $msg_obj->subject;
	$subject = "Re: $subject" if (defined $subject and $subject !~ /^Re:/i);
    }
    return $subject;
}


##
# $self->message;
# ---------------
#   Returns the default message for this message.
##
sub message
{
    my $self = shift;
    my $cgi  = cgi();
    my $message = $cgi->param ('message');
    return $message if (defined $message and $message);
    
    my $msg_obj = $self->reply_message;
    return '' unless (defined $msg_obj);
    return join "\n", map { (defined $_) ? "> $_" : () } split /\n/sm, $msg_obj->body;
}


##
# $self->body;
# ------------
#   Returns the body for this message;
##
sub body
{
    my $self = shift;
    my $cgi  = cgi();
    my $body = $cgi->param ('body');
    return $body if (defined $body and $body);
    
    my $msg_obj = $self->reply_message;
    my @lines = split /(?:\n|\r)+/sm, $msg_obj->body;
    return join "\n", map { "> $_" } @lines;
}


##
# $class->langs;
# --------------
#   Returns a structure which can be used within a template to choose
#   a document cache control value from a select box.
##
sub langs
{
    my $self = shift;
    my $cgi  = cgi();
    my $doc  = $self->parent();
    
    my $current_lang = undef;
    if (defined $cgi->param ('lang'))      { $current_lang = $cgi->param ('lang')           }
    elsif (defined $self->reply_message()) { $current_lang = $self->reply_message()->lang() }
    else                                   { $current_lang = $doc->lang()                   }
    
    my $languages_rtl = new MKDoc::Config ('languages_rtl');
    my @res = map {
	{
	    value          => $_,
	    label          => $doc->lang_label ($_),
	    selected       => ($_ eq $current_lang),
	    direction      => ($languages_rtl->get ($_)) ? 'rtl'   : 'ltr',
	    align          => ($languages_rtl->get ($_)) ? 'right' : 'left',
	    align_opposite => ($languages_rtl->get ($_)) ? 'left'  : 'right',
	}
    } $doc->available_langs;
    
    return wantarray ? @res : \@res;
}


##
# $self->selected_cache_control;
# ------------------------------
#   Returns the cache_control which is currently selected
##
sub selected_lang
{
    my $self = shift;
    my ($selected) = map { ($_->{selected}) ? $_ : () } $self->langs (@_);
    $selected ||= do {
	my $languages_rtl = new MKDoc::Config ('languages_rtl');
	$_ = $self->parent->lang;
	{
	    value => $_,
	    label => $self->parent->lang_label,
	    selected => 'selected',
	    direction      => ($languages_rtl->get ($_)) ? 'rtl'   : 'ltr',
	    align          => ($languages_rtl->get ($_)) ? 'right' : 'left',
	    align_opposite => ($languages_rtl->get ($_)) ? 'left'  : 'right',
	}
    };
    return $selected;
}


##
# $self->unselected_langs;
# ------------------------
#   Returns the langs which are currently not selected
##
sub unselected_langs
{
    my $self = shift;
    $self = ref $self || $self;
    my @unselected = map { ($_->{selected}) ? () : $_ } $self->langs (@_);
    return (wantarray) ? @unselected : \@unselected;
}


##
# $self->post_errors;
# -------------------
#   Returns a hashref of errors if any.
##
sub post_errors
{
    my $self = shift;
    return unless ($ENV{REQUEST_METHOD} =~ /^POST$/i);
    
    my $cgi = cgi();
    my $error_nb = undef;
    my $subject  = $cgi->param ('subject');
    my $language = $cgi->param ('language') || 'en';
    my $message  = $cgi->param ('message');

    my %error = ();
    $error{subject}  = 1 unless (defined $subject and $subject ne '');
    $error{message}  = 1 unless (defined $message and $message ne '');
    $error{language} = 1;
    
    if (defined $language and $language ne '')
    {
	foreach my $lang ($self->langs)
	{
	    if ($lang->{value} eq $language)
	    {
		my $tmp = delete $error{language};
		last;
	    }
	}
    }
    
    return unless (scalar (keys %error));
    return \%error;
}


# yet another horrible hack
sub ancestors
{
    my $self = shift;
    my @res  = ($self->SUPER::ancestors, $self->component);
    return wantarray ? @res : \@res;
}


##
# $self->http_get;
# ----------------
#   Processes a GET request.
##
sub http_get
{
    my $self = shift;
    my $cgi = cgi();
    my $header = new flo::HTTP::Header;
    $header->set ('Content-Type: text/html; charset=UTF-8');
    $self->render_http (
	__input__  => 'XML',
	__output__ => 'XHTML',
	self       => $self,
	object     => $self->component,
       );
}


##
# $self->http_post;
# -----------------
#   Processes a POST request.
##
sub http_post
{
    my $self = shift;
    my $cgi = cgi();
    return $self->http_get if (defined $self->post_errors);
    
    my $msg_obj = $self->reply_message;
    my $reply_id   = $msg_obj->message_id if (defined $msg_obj);
    my $references = $msg_obj->references if (defined $msg_obj);
    chomp ($references) if (defined $references);
    
    # post the message to the imap folder...
    my $subject  = $self->subject;
    my $message  = $self->message;
    my $language = $cgi->param ('lang') || $cgi->param ('language');
    
    my $imap = new flo::plugin::Discussion::IMAP();
    if (defined $reply_id and defined $references)
    {
	$reply_id =~ s/^\<*//sm;
	$reply_id =~ s/\>*$//sm;
	$imap->post ( {
	    realname    => $self->user()->first_name . " " . $self->user()->last_name,
	    email       => $self->user()->email,
	    subject     => $subject,
	    message     => $message,
	    language    => $language,
	    in_reply_to => "<$reply_id>",
	    references  => "$references <$reply_id>",
	} );
    }
    elsif (defined $reply_id)
    {
	$reply_id =~ s/^\<*//sm;
	$reply_id =~ s/\>*$//sm;
	$imap->post ( {
	    realname    => $self->user()->first_name . " " . $self->user()->last_name,
	    email       => $self->user()->email,
	    subject     => $subject,
	    message     => $message,
	    language    => $language,
	    in_reply_to => "<$reply_id>",
	    references  => "<$reply_id>",
	} );
    }
    else
    {
	$imap->post ( {
	    realname    => $self->user()->first_name . " " . $self->user()->last_name,
	    email       => $self->user()->email,
	    subject     => $subject,
	    message     => $message,
	    language    => $language,
	} );
    }
    
    my $component = $self->component;
    print cgi()->redirect ($component->uri);
}


sub uri
{
    my $self = shift;
    my $uri  = $self->SUPER::uri();
    $uri .= "," . MKDoc::Config->IMAP_POST_URI_HINT;
    return $uri;
}


1;


__END__
