# ------------------------------------------------------------------------
# flo::plugin::Sitemap
# ------------------------------------------------------------------------
# Author : Sam Tregar.
# Copyright : (c) MKDoc Holdings Ltd, 2005.
#
# This file is part of MKDoc. 
# 
# MKDoc is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# MKDoc is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with MKDoc; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ------------------------------------------------------------------------
package flo::plugin::Sitemap;
use strict;
use warnings;

=head1 NAME

flo::plugin::Sitemap - a sitemap which shows nearby documents only

=head1 SYNOPSIS

Access this plugin like this:

  http://www.example.com/.sitemap.html
  http://www.example.com/foo/.sitemap.html

=head1 DESCRIPTION

This plugin implements a scope-limited sitemap for a given document.
Links are shown for all ancestores, direct children and siblings.  If
the user has permission to visit a document it will be a link.  If it
has children of its own it will show a [+].  Hidden documents are
shown only to editors.

=cut

use flo::Standard;
use lib::sql::Condition;
use base qw /flo::Plugin/;

sub _name_default { '.sitemap.html' }

sub run {
    my $self = shift;
    $self->render_http (
      self       => $self,
      __input__  => 'XML',
      __output__ => 'XHTML',
       );

    return 'TERMINATE';
}

sub template_path { '/sitemap' }

sub nodes {
    my $self = shift;
    my $doc  = flo::Standard::current_document();
    $self->{_current_doc} = $doc;

    my ($tree, $parent, $cur) = $self->ancestor_tree($doc);

    # add nodes for siblings if there is a parent (i.e. not for root)
    $self->add_children($parent, $doc->parent) if $parent;

    # add child nodes for the current doc
    $self->add_children($cur, $doc);

    return $tree;
}

# adds child nodes, sorting based on document sort ordering
sub add_children {
    my ($self, $node, $doc) = @_;
    my $cur = $self->{_current_doc};
    
    my @nodes = map { $self->new_node($_) } 
                grep { $_->id != $cur->id }
                $self->get_children($doc);
    push @{$node->{children}}, @nodes;

    # sort the sibling list
    $node->{children} = $self->sort_children($doc, $node->{children});
}

sub sort_children {
    my ($self, $doc, $nodes) = @_;

    my $sort_on = $doc->{Sort_By};

    my @result;
    if ($sort_on eq 'Sibling_Position') {
	@result = sort { $a->{doc}{$sort_on} <=> $b->{doc}{$sort_on} } @$nodes;
    } else {
	@result = sort { $a->{doc}{$sort_on} cmp $b->{doc}{$sort_on} } @$nodes;
    }

    @result = reverse @result if $doc->{Order_By};

    return \@result;
}

sub ancestor_tree {
    my ($self, $doc) = @_;

    # build nodes for the ancestors and the current doc
    my @nodes = map { $self->new_node($_) } ($doc->ancestors, $doc);

    # build the tree, top down
    my $head = (my $p = shift @nodes);
    my $last;
    foreach my $node (@nodes) {
        $last = $p;
        $p->{children} = [ $node ];
        $p = $p->{children}[0];
    }

    # return the root and pointer to the current doc and its parent
    return ($head, $last, $p);
}

# creates a new node hash for use in the template, takes a document as
# the sole arg
sub new_node {
    my ($self, $doc) = @_;
    my $children = $self->get_children($doc);
    return { 
            doc          => $doc,
            has_children => @$children ? 1 : 0,
            is_selected  => $doc->id == $self->{_current_doc}->id,
           };
}

# fetch children from a document, allowing editors to see hidden documents
sub get_children {
    my ($self, $doc) = @_;
    return $doc->children_showable() 
      unless $self->user and $self->user->group eq 'editor';

    # filter out unshowable but keep the hidden ones
    my @children = grep { $_->is_showable or $_->is_hidden } $doc->children();
    
    return wantarray ? @children : \@children;
}

1;
