#!/usr/bin/perl
package MKDoc::Misc::Newsletter;
use strict;
use warnings;
use Text::Wrap;
use MKDoc;
use flo::Record::Document;
use Encode;
use base qw /flo::Plugin/;
use Petal::Mail;
use Carp qw(croak);

sub template_path { '/newsletter' }

# keep caches to avoid repeated lookups of group permissions info.
# These could be disabled if the extra memory used is too much but it
# seems unlikely to get huge.
our %DOCUMENT_GROUP_CACHE;
our %USER_GROUP_CACHE;

sub root
{
    my $document_t = flo::Standard::table ('Document');
    return $document_t->get (1);
}


sub user
{
    my $self = shift;
    return $self->{user};
}


sub send 
{
    my $self = shift;
    my $mail = new Petal::Mail (
        file => '/newsletter',
        language => $self->language() || 'en',
    );
    
    $mail->send (
	self       => $self,
	__input__  => 'XML',
	__output__ => 'XML',
    );
    
    return 1; 
}

# returns true if there are newest documents to include in the newsletter
sub has_newest {
    my $self = shift;
    my $res = $self->results("newest");
    return 1 if @$res;
    return 0;
}

# returns true if there are upcoming events to include in the newsletter
sub has_upcoming {
    my $self = shift;
    my $res = $self->results("upcoming");
    return 1 if @$res;
    return 0;
}

# return ref to array of hashes containing newsletter results.  Takes
# a single parameter specifying the mode, either "newest" or
# "upcoming".  This method cache the results for each mode.
sub results {
    my $self = shift;    
    my $mode = shift || croak("Missing required mode parameter");

    # return cached data if available
    return $self->{"_cache_$mode"} if $self->{"_cache_$mode"};

    my $user_login = $self->user()->login();

    # concoct SQL needed for upcoming or newest mode
    my ($extra_from, $extra_where, $extra_select, @extra_params,
        $order_by, $group_by);
        
    if ($mode eq 'upcoming') {
        # setup for selecting by timerange
        $order_by     = "Document_TimeRange.FromDate ASC";
        $group_by     = "Document_TimeRange.ID";
        $extra_from   = ", Document_TimeRange";
        $extra_select = ", Document_TimeRange.ID as Document_TimeRange_ID";

        # select documents with timeranges dates from now to the end
        # of the range
        $extra_where  = <<END;
          AND Document.ID = Document_TimeRange.Document_ID
          AND ((Document_TimeRange.FromDate >= ? AND 
                Document_TimeRange.FromDate <= ?)
               OR 
               (Document_TimeRange.ToDate >= ? AND 
                Document_TimeRange.ToDate <= ?))
END
        @extra_params = ($self->compute_date_now(), 
                         $self->compute_date_forward()) x 2;
    } else {
        # select documents created in the chosen period
        $order_by     = "Date_Created DESC";
        $group_by     = "Document.ID";
        $extra_where  = "AND Document.Date_Created > ?";
        ($extra_from, $extra_select) = ("") x 2;
        @extra_params = $self->compute_date_since();
    }

    my $sql  = <<SQL;
SELECT Document.ID AS ID, SUM(Preference_Audience.Value) AS Pref_Score,
       Creator.First_Name as Creator_First_Name,
       Creator.Family_Name as Creator_Family_Name
       $extra_select
FROM   Document, Document_Audience, Audience, Preference_Audience, Editor, Editor as Creator, Preference_Language
       $extra_from
WHERE
        -- join the tables together
        (
                Preference_Language.Language_ID = Document.Lang         AND
                Preference_Language.Editor_ID = Editor.ID               AND
                Preference_Audience.Audience_ID = Audience.ID           AND
                Preference_Audience.Editor_ID = Editor.ID               AND
                Document_Audience.Audience_ID = Audience.ID             AND
                Document_Audience.Document_ID = Document.ID		AND
		Document.Editor_Created_ID = Creator.ID
        )
AND
        -- limit to the current editor
        Editor.Login = ?
AND
        -- remove languages which are not wanted
        Preference_Language.Value = 1
$extra_where


GROUP BY $group_by
ORDER BY $order_by
SQL

    my $dbh = lib::sql::DBH->get();
    my $sth = $dbh->prepare_cached ($sql);
    $sth->execute ($user_login, @extra_params);
    
    my @res;
    my $doc_t = flo::Standard::table ('Document');
    ROW: while (my $h = $sth->fetchrow_hashref) {
        next unless ($h->{Pref_Score} > 0);
	my $doc = $doc_t->get ( $h->{ID} );
	next unless ($doc->is_showable());

        # check permissions (this can't go in the SQL above due to the
        # tree-walking needed to test a document's group membership)
        next unless user_can_see($self->user, $doc);
        
	$Text::Wrap::columns = 72;
	$Text::Wrap::columns = 72;

        # assemble data for template from the document object and the
        # query results
        my %row;
        $row{doc}           = $doc;
        ($row{Full_Path}    = $doc->{Full_Path}) =~ s/^\//$::PUBLIC_DOMAIN/;
        ($row{Date_Created} = $doc->date_created) =~ s/\s+.*//;
        $row{Description}   = join '',fill ('    ', '    ', $doc->description);
        $row{Title}         = $doc->title;
        $row{Creator_First_Name}  = $h->{Creator_First_Name};
        $row{Creator_Family_Name} = $h->{Creator_Family_Name};

        # deal with timerange data for upcoming events list
        if ($mode eq 'upcoming') {
            # find this timerange in the list of components for this doc
            my $timerange_id = $h->{Document_TimeRange_ID};
            my ($timerange) = 
              grep { $_->isa('flo::editor::TimeRange') and
                     $_->Document_TimeRange_ID == $timerange_id }
                $doc->components;
            # ignore hits with no matching timerange.
            next ROW unless $timerange;

            $row{TimeRange_Title}    = $timerange->title;
            $row{TimeRange_FromDate} = $timerange->from_datetime;
            $row{TimeRange_ToDate}   = $timerange->to_datetime;
        }

	push @res, \%row;
    }
    
    # save for future requests
    return $self->{"_cache_$mode"} = \@res;
}

# determine if user should be able to see this document, looking at
# group permissions.  Returns 1 if the user has access, 0 if not.
sub user_can_see {
    my ($user, $document) = @_;

    # lookup groups for the document, no groups means everyone can see it
    my @doc_group_ids = find_groups($document);
    return 1 unless @doc_group_ids;

    # get a list of the user's groups
    my @groups = user_groups($user);

    # no results means the user wasn't in any of the groups, denied
    return 0 unless @groups;

    # allow through if the user is in one of the document's groups
    my %groups = map { ($_->{Grp_ID}, 1) } @groups;
    return 1 if grep { $groups{$_} } @doc_group_ids;
    
    # no dice
    return 0;
}

# get a list of groups for a particular user
sub user_groups {
    my $user = shift;
    return @{$USER_GROUP_CACHE{$user->id}} 
      if $USER_GROUP_CACHE{$user->id};

    my $editor_grp_t = flo::Standard::table('Editor_Grp');
    my $con          = lib::sql::Condition->new(Editor_ID => $user->id);
    my @groups       = $editor_grp_t->select(cols  => 'Grp_ID',
                                             where => $con)->fetch_all;

    $USER_GROUP_CACHE{$user->id} = \@groups;
    return @groups;
}

# get groups for a document, looking up the tree
sub find_groups {
    my $document = shift;
    my $document_grp_t = flo::Standard::table('Document_Grp');

    return @{$DOCUMENT_GROUP_CACHE{$document->id}} 
      if $DOCUMENT_GROUP_CACHE{$document->id};

    # get list of all documents to check
    my @documents = ($document, $document->ancestors);

    # get results for each document 
    my %groups;
    foreach my $doc (@documents) {
        my @res = $document_grp_t->select (
	cols => 'Grp_ID',
        where => lib::sql::Condition->new(Document_ID => $doc->id)
                                      )->fetch_all();
        $groups{$_->{Grp_ID}} = 1 for @res;
    }

    $DOCUMENT_GROUP_CACHE{$document->id} = [keys %groups];
    return @{$DOCUMENT_GROUP_CACHE{$document->id}};
}

# translate a time to day-month-year for use with MySQL
sub _time2date {
    my $self = shift;
    my ($mday, $mon, $year) = (localtime(shift))[3,4,5];
    $mon += 1;
    $year += 1900;
    return "$year-$mon-$mday";
}

sub compute_date_now { shift->_time2date(time()) }

package MKDoc::Misc::Newsletter::Daily;
use strict;
use warnings;
our @ISA = qw /MKDoc::Misc::Newsletter/;

sub compute_date_since { shift->_time2date(time - 24 * 3600) }

sub compute_date_forward { shift->_time2date(time + 24 * 3600) . " 23:59:59" }

sub is_daily { 1 }

package MKDoc::Misc::Newsletter::Weekly;
use strict;
use warnings;
our @ISA = qw /MKDoc::Misc::Newsletter/;

sub compute_date_since { shift->_time2date(time - 7 * 24 * 3600) }

sub compute_date_forward { shift->_time2date(time + 7 * 24 * 3600) . 
                             " 23:59:59" }

sub is_weekly { 1 }


package MKDoc::Misc::Newsletter::Monthly;
use strict;
use warnings;
our @ISA = qw /MKDoc::Misc::Newsletter/;

sub compute_date_since {
    my $self = shift;
    my ($mday, $mon, $year) = (localtime(time))[3,4,5];
    $year += 1900;
    ($mon == 0) and do {
	$year--;
	$mon = 12;
    };
    
    return "$year-$mon-$mday";
}

sub compute_date_forward {
    my $self = shift;
    my ($mday, $mon, $year) = (localtime(time))[3,4,5];
    $year += 1900;
    $mon  += 2;
    ($mon == 13) and do {
	$year++;
	$mon = 1;
    };
    
    return "$year-$mon-$mday 23:59:59";
}

sub is_monthly { 1 }



package main;
use strict;
use warnings;
use Carp;
use MKDoc::Handler::Initialize;
use Text::Wrap;

use constant MONTHLY => 'newsletter-monthly';
use constant WEEKLY  => 'newsletter-weekly';
use constant DAILY   => 'newsletter-daily';

our $PERIODICITY;
our $DBH;

$::PUBLIC_DOMAIN ||= $ENV{PUBLIC_DOMAIN} || die 'PUBLIC_DOMAIN is not defined';
 

MKDoc::Handler::Initialize::handler();

main();


sub main
{
    my $newsletter  = undef;
    my $periodicity = periodicity();
    
    print "Sending newsletter with periodicity: $periodicity\n"; 
    foreach my $user (users_which_want_newsletter())
    {
	print "\n\t$user->{Login}";
	my $newsletter = instantiate_newsletter ($user) || next;
        $newsletter->send();
        print "\t$user->{First_Name} $user->{Family_Name} <$user->{Email}>";
    }
}


sub instantiate_newsletter
{
    my $user = shift;
    my $periodicity = periodicity();
    my %args = ( user => $user );
    my $news = undef;
    ($periodicity eq DAILY)   and $news = new MKDoc::Misc::Newsletter::Daily (%args);
    ($periodicity eq WEEKLY)  and $news = new MKDoc::Misc::Newsletter::Weekly (%args);
    ($periodicity eq MONTHLY) and $news = new MKDoc::Misc::Newsletter::Monthly (%args);
    
    # don't send a newsletter if there's no news
    return unless $news->has_upcoming or $news->has_newest;
    return $news;
}


sub periodicity
{
    $PERIODICITY ||= do {
	my $res = undef;
	$_ = shift (@ARGV) || 'none';
	/^(day|week|month)$/ or die "Usage: $0 (day|week|month)";
	/day/   and $res = DAILY;
	/week/  and $res = WEEKLY;
	/month/ and $res = MONTHLY;
	$res;
    };
    
    return $PERIODICITY;
}


sub users_which_want_newsletter
{
    my $sql = join "\n",
        qq |SELECT DISTINCT Editor.*                 |,
	qq |FROM   Editor, Preference                |,
	qq |WHERE                                    |,
	qq |    Preference.Editor_ID = Editor.ID AND |,
	qq |    Preference.Name = ?              AND |,
	qq |    Preference.Value > 0                 |;
    
    my $dbh = lib::sql::DBH->get();
    my $sth = $dbh->prepare_cached ($sql);
    $sth->execute (periodicity());

    my $query = new lib::sql::Query
        sth => $sth,
	bless_into => 'flo::Record::Editor';
    
    my @res = $query->fetch_all();
    @res = map { $_->is_confirmed() ? $_ : () } @res;
    return wantarray ? @res : \@res;
}


1;


__END__
